Start-Transcript "$($MyInvocation.MyCommand.Path).$((Get-Date).ToString('yyyy-MM')).log" -Append -IncludeInvocationHeader

$scriptName = 'Set-LocalAdminPassword'
$managementScripts = "$env:ProgramData\ManagementScripts"
$scriptPath = "$managementScripts\$scriptName.ps1"

$taskPath = "\ManagementScripts\"

# Check if the scheduled task exists
$scheduledTaskExists = Get-ScheduledTask -TaskName $scriptName -TaskPath $taskPath -ErrorAction SilentlyContinue
if ($scheduledTaskExists) {
    # Unregister the scheduled task
    Unregister-ScheduledTask -TaskName $scriptName -TaskPath $taskPath -Confirm:$false
    Write-Host "Scheduled Task removed successfully."
} else {
    Write-Host "Scheduled Task does not exist."
}

# Check if the script file exists
if (Test-Path -Path $scriptPath) {
    # Delete the script file
    Remove-Item -Path $scriptPath -Force
    Write-Host "Script file removed successfully."
} else {
    Write-Host "Script file does not exist."
}

$ErrorActionPreference = 'Stop'
$accountName = 'devicelocaladmin'

try {
    $pw = Get-BitLockerVolume -MountPoint C: | % { $_.KeyProtector.RecoveryPassword } | ? { $_ } | Select -Last 1
    if ($pw) { 
        Write-Host "Using BitLocker key ending in $($pw.Substring([Math]::Max(0, $pw.Length - 6))) for password." 
        $pw = (($pw -split '-' | Select-Object -First 3) -join '-') + 'X!@' 			
    }
} 
catch {
}

if (!$pw) {
    # if there is no BitLocker password or an exception is thrown
    Write-Host "BitLocker key not available. Skipping this execution."
    return
}

$encryptedPassword = ConvertTo-SecureString $pw -AsPlainText -Force

#Make sure that the "the user's password must be changed before signing" in option is not enabled, if so we delete the account
$user = get-localuser -Name $accountName -ErrorAction SilentlyContinue
if (!$user.PasswordExpires) {
    Write-Host "PasswordExpires value is empty, delete the account"
    remove-localuser -Name $accountName -ErrorAction SilentlyContinue
} else {
    Write-Host "PasswordExpires value is available, continue"
}

$user = get-localuser -Name $accountName -ErrorAction SilentlyContinue
if (!$user) {
    Write-Host "Creating local user $accountName."
    $user = New-LocalUser -Name $accountName -FullName $accountName `
        -AccountNeverExpires `
        -Description 'Device local computer admin account' `
        -PasswordNeverExpires -Password $encryptedPassword
}

$memberOfLocalAdmins = ([ADSI]"WinNT://./Administrators").psbase.Invoke('Members') | % { ([ADSI]$_).InvokeGet('AdsPath') } | ? { $_.EndsWith("$($env:COMPUTERNAME)/$accountName") }
if (!$memberOfLocalAdmins) {
    Write-Host "Adding $accountName to local Administrators group."
    Add-LocalGroupMember -Group 'Administrators' -Member $user
}

$registryKey = "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\LAPS\State"

if (Test-Path -Path $registryKey) {
    Write-Host "Registry key $registryKey exists. Skipping password change."
} else {
    # Make sure the password is the expected one and reset it if not
    Add-Type -AssemblyName System.DirectoryServices.AccountManagement
    $ds = New-Object System.DirectoryServices.AccountManagement.PrincipalContext('machine', $env:COMPUTERNAME)
    if (!$ds.ValidateCredentials($accountName, $pw)) {
        Write-Host "$accountName's password doesn't match expected; resetting password."
        $user | Set-LocalUser -PasswordNeverExpires:$true -UserMayChangePassword:$true -Password $encryptedPassword
    }
    $ds.Dispose()
}

Write-Host "Done configuring local administrator user $accountName."

Stop-Transcript
